# Step 4: Simulation Results ----

# Prep ----
# Load packages
library(lavaan)
library(magrittr)
library(dplyr)
library(tidyr)
library(ggplot2)
library(effectsize)

# Load simulated data
sim_data <- readRDS("data/simdata_revision.RDS")

# Load simulation conditions
sim_conditions <- readRDS("data/simconditions.RDS")

# Get simulation results
sim_results <- readRDS("results/simresults_default.RDS")

sim_results <- sim_results %>% 
  mutate(err = as.factor(err)) %>%
  mutate_if(is.character, as.numeric)

# Load Integer identification function
source("../altcoding.R")

# Figure out missing iterations ----
sim_results %>% filter(!is.na(err)) %>%
  group_by(condition_number) %>%
  summarize(n = n())

# Set up data frames for tables/figures ----

# Create Data Frame for convergence figures
sim_plotdata <- sim_conditions %>%
  mutate(condition_number = 1:nrow(sim_conditions)) %>%
  full_join(sim_results)  %>%
  select(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, 
         condition_number, iter, contains("conv"), contains("admis")) %>%
  filter(resp_options != 2) %>%
  pivot_longer(cols = contains(c("conv", "admis")), 
               names_to = c("model", "status"), 
               names_pattern = "(.*)_(.*)", 
               values_to = "value") %>%
  pivot_wider(names_from = status, values_from = value) %>%
  mutate(status = case_when(is.na(conv) ~ "Not converged",
                            conv == 0 ~ "Not converged",
                            conv == 1 & admis == 0 ~ "Converged but not admissible",
                            conv == 1 & admis == 1 ~ "Converged and admissible"),
         identification = if_else(model == "i1", 1, if_else(model == "i2", 2, 3))) %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, identification, status) %>%
  summarize(freq = n()) %>% 
  mutate(prop = freq / 500)

# Create Data Frame for ANOVA
sim_anovadata <- sim_plotdata %>% filter(status == "Converged and admissible") %>%
  mutate(identification = factor(identification, 
                                 levels = c(1,3,2), 
                                 labels = c("Integer", "Unit Variance", "Reference Indicator")))

# Create Data Frame for Chi-square Fit Index Comparison
sim_fitres <- sim_conditions %>%
  mutate(condition_number = 1:nrow(sim_conditions)) %>%
  full_join(sim_results)  %>%
  mutate(rowid = row_number()) %>%
  select(rowid, resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, 
         condition_number, iter, contains("conv"), contains("admis"), contains("chisq")) %>%
  filter(resp_options != 2) %>%
  mutate(i1_chisq = as.numeric(i1_chisq),
         i2_chisq = as.numeric(i2_chisq),
         i3_chisq = as.numeric(i3_chisq)) %>%
  pivot_longer(cols = c("i1_conv", "i1_admis","i1_chisq",
                        "i2_conv", "i2_admis", "i2_chisq",
                        "i3_conv", "i3_admis", "i3_chisq"), 
               names_to = c("model", "status"), 
               names_pattern = "(.*)_(.*)", 
               values_to = "value") %>%
  pivot_wider(names_from = status, values_from = value) %>%
  mutate(status = case_when(is.na(conv) ~ "Not converged",
                            conv == 0 ~ "Not converged",
                            conv == 1 & admis == 0 ~ "Converged but not admissible",
                            conv == 1 & admis == 1 ~ "Converged and admissible"),
         identification = if_else(model == "i1", 1, if_else(model == "i2", 2, 3)))

# Run ANOVA ----
fit <- lm(freq ~ resp_options + resp_pattern + num_ind + prop_sparse + loading_mag + identification, 
          data = sim_anovadata)

petad <- effectsize(anova(fit))
petad

# Create Figure showing breakdown in convergence status ----
p1d <- sim_plotdata %>%
  filter(num_ind == 3 & resp_pattern == "sparse1" & prop_sparse == 1) %>%
  mutate(freq2 = if_else(status == "Converged and admissible", freq, NA)) %>%
  ggplot(aes(x = identification, y = freq, fill = status)) +
  geom_bar(position="stack", stat="identity") +
  geom_text(aes(label = freq2), position = position_stack(0.5), family = "serif") +
  ggtitle("Population model: 3 factor model with 3 indicators each,\nof which 100% have a sparse response pattern") +
  scale_x_continuous("Identification Method", 
                     breaks = 1:3, 
                     labels = c("Integer", "Reference-Indicator", "Unit-Variance"),
                     sec.axis = sec_axis(~ . , name = "Response categories", breaks = NULL, labels = NULL)) +
  scale_y_continuous("Frequency",
                     sec.axis = sec_axis(~ . , name = "Factor Loading Magnitude", breaks = NULL, labels = NULL)) +
  scale_fill_grey("", start = .5, end = .9) +
  facet_grid(rows = vars(loading_mag), cols = vars(resp_options)) +
  jtools::theme_apa() +
  theme(text = element_text(family = "serif"),
        axis.text.x = element_text(angle = 30, hjust = 1),
        legend.position = "bottom",
        axis.title = element_text(face = "bold"))

# Create figure showing converged and admissible across conditions ----
p2d <- sim_plotdata %>%
  mutate(identification = factor(identification, 
                                 levels = c(1,3,2), 
                                 labels = c("Integer", "Unit-Variance", "Reference-Indicator"))) %>%
  filter(status == "Converged and admissible" & ((resp_pattern %in% c("sparse1", "sparse2") & prop_sparse == 1) | resp_pattern == "balanced")) %>%
  ggplot(aes(x = loading_mag, y = prop, 
             group = interaction(identification, resp_pattern)), 
         alpha = .6) +
  geom_point(aes(color = identification, 
                 shape = identification)) + 
  geom_line(aes(color = identification, 
                linetype = resp_pattern)) + 
  scale_x_continuous("Loading magnitude", 
                     breaks = c(0.4, 0.6, 0.8), 
                     sec.axis = sec_axis(~ . , 
                                         name = "Response categories", 
                                         breaks = NULL, labels = NULL)) +
  scale_y_continuous("Proportion converged and admissible",
                     sec.axis = sec_axis(~ . , 
                                         name = "Number of indicators per factor", 
                                         breaks = NULL, labels = NULL)) +
  scale_shape_manual("Identification:", values = c(1,2,7)) +
  scale_color_grey("Identification:", start = .8, end = .2) +
  scale_linetype_manual("Response Pattern:",
                        values = c(1,2,3),
                        breaks = c("balanced", "sparse1","sparse2"),
                        labels = c("Symmetric", "Skewed", "Middling")) +
  facet_grid(rows = vars(num_ind), cols = vars(resp_options)) +
  jtools::theme_apa() +
  theme(text = element_text(family = "serif"),
        axis.text.x = element_text(angle = 30, hjust = 1),
        legend.position = "bottom",
        legend.box="vertical",
        legend.title = element_text(face = "bold", size = 12),
        axis.title = element_text(face = "bold"))

save(p1d, p2d, petad, file = "figures/manuscript_revision_default_figures.RData")

# Create tables for Chi-square fit differences ----

# Explore overall results
sim_fitres %>% 
  mutate(chisq = round(chisq, 5)) %>%
  select(iter, resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, model, status, chisq) %>%
  pivot_wider(names_from = "model", values_from = c("chisq", "status")) %>%
  filter(status_i1 == "Converged and admissible" & status_i3 == "Converged and admissible" & status_i2 == "Converged and admissible") %>%
  mutate(same = if_else(chisq_i1 == chisq_i2 & chisq_i1 == chisq_i3 & chisq_i2 == chisq_i3, "same", "different")) %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag) %>%
  summarize(psame = sum(same == "same") / n()) %>%
  pivot_wider(names_from = c("num_ind", "resp_options"), values_from = psame) %>%
  print(n = 21)

# Create Table with prop Chi-square differences for Manuscript
addtorow1d <- list()
addtorow1d$pos <- list(0,0, 0)
addtorow1d$command <- c("Loading Mag. & Prop. Sparse &  \\multicolumn{6}{c}{Response Options}\\\\",
                       "& & \\multicolumn{3}{c}{3 Indicators} & \\multicolumn{3}{c}{6 Indicators}\\\\",
                       "& & 3 & 4 & 5 & 3 & 4 & 5\\\\")

tab_chisq1 <- sim_fitres %>% 
  mutate(chisq = round(chisq, 5)) %>%
  select(iter, resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, model, status, chisq) %>%
  pivot_wider(names_from = "model", values_from = c("chisq", "status")) %>%
  filter(status_i1 == "Converged and admissible" & status_i3 == "Converged and admissible" & status_i2 == "Converged and admissible") %>%
  mutate(same = if_else(chisq_i1 == chisq_i2 & chisq_i1 == chisq_i3 & chisq_i2 == chisq_i3, "same", "different"),
         prop_sparse= if_else(prop_sparse==.66,.67, prop_sparse)) %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag) %>%
  summarize(psame = sum(same == "same") / n()) %>%
  pivot_wider(names_from = c("num_ind", "resp_options"), values_from = psame) %>%
  filter(resp_pattern %in% c("sparse2")) %>%
  ungroup() %>%
  select(-resp_pattern) %>%
  select(loading_mag,prop_sparse, `3_3`,`3_4`,`3_5`,`6_3`,`6_4`,`6_5`)%>%
  arrange(loading_mag)

tab_chisq1$loading_mag <- ifelse(duplicated(tab_chisq1$loading_mag),"",tab_chisq1$loading_mag)

tab_chisq1d <- tab_chisq1

# Add fancy header columns to table
addtorow2d <- list()
addtorow2d$pos <- list(0,0, 0)
addtorow2d$command <- c("Pattern & Loading Mag. & Prop. Sparse & \\multicolumn{6}{c}{Response Options}\\\\",
                      "& & & \\multicolumn{3}{c}{3 Indicators} & \\multicolumn{3}{c}{6 Indicators}\\\\",
                      "& & &  3 & 4 & 5 &  3 & 4 & 5\\\\")

# Create Table with prop Chi-square differences for Appendix
tab_chisq2 <- sim_fitres %>% 
  mutate(chisq = round(chisq, 5)) %>%
  select(iter, resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, model, status, chisq) %>%
  pivot_wider(names_from = "model", values_from = c("chisq", "status")) %>%
  filter(status_i1 == "Converged and admissible" & status_i3 == "Converged and admissible" & status_i2 == "Converged and admissible") %>%
  mutate(same = if_else(chisq_i1 == chisq_i2 & chisq_i1 == chisq_i3 & chisq_i2 == chisq_i3, "same", "different"),
         prop_sparse= if_else(prop_sparse==.66,.67, prop_sparse)) %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag) %>%
  summarize(psame = sum(same == "same") / n()) %>%
  pivot_wider(names_from = c("num_ind", "resp_options"), values_from = psame) %>%
  filter(resp_pattern %in% c("balanced", "sparse1")) %>%
  mutate(resp_pattern = if_else(resp_pattern == "balanced", "Symmetric", "Skewed"),
         resp_pattern = factor(resp_pattern, levels = c("Symmetric","Skewed"))) %>%
  select(resp_pattern, loading_mag, prop_sparse,`3_3`,`3_4`,`3_5`,`6_3`,`6_4`,`6_5`)%>%
  arrange(resp_pattern, loading_mag) %>%
  mutate(resp_pattern = as.character(resp_pattern))

tab_chisq2$resp_pattern <- ifelse(duplicated(tab_chisq2$resp_pattern),"",tab_chisq2$resp_pattern)
tab_chisq2$loading_mag[4:nrow(tab_chisq2)] <- ifelse(duplicated(tab_chisq2$loading_mag[4:nrow(tab_chisq2)]),"",tab_chisq2$loading_mag[4:nrow(tab_chisq2)])

tab_chisq2d <- tab_chisq2

# Figure out which identification results in different/best fit
which_diff <- sim_fitres %>% 
  mutate(chisq = round(chisq, 5)) %>%
  select(rowid, iter, resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, model, status, chisq) %>%
  pivot_wider(names_from = "model", values_from = c("chisq", "status")) %>%
  filter(status_i1 == "Converged and admissible" & status_i3 == "Converged and admissible" & status_i2 == "Converged and admissible") %>%
  mutate(same = if_else(chisq_i1 == chisq_i2 & chisq_i1 == chisq_i3 & chisq_i2 == chisq_i3, "same", "different")) %>%
  mutate(whichdiff = case_when(chisq_i1 == chisq_i2 & chisq_i1 != chisq_i3 ~ "3",
                               chisq_i1 == chisq_i3 & chisq_i1 != chisq_i2 ~ "2",
                               chisq_i2 == chisq_i3 & chisq_i1 != chisq_i3 ~ "1",
                               chisq_i1 != chisq_i2 & chisq_i1 != chisq_i3 & chisq_i2 != chisq_i3 ~ "all",
                               same == "same" ~ "none"),
         minchisq = pmin(chisq_i1, chisq_i2, chisq_i3),
         whichmin = case_when(minchisq == chisq_i1 & minchisq == chisq_i2 & minchisq == chisq_i3 ~ "1, 2, 3",
                              minchisq == chisq_i1 & minchisq == chisq_i2 ~ "1, 2",
                              minchisq == chisq_i1 & minchisq == chisq_i3 ~ "1, 3",
                              minchisq == chisq_i2 & minchisq == chisq_i3 ~ "2, 3",
                              minchisq == chisq_i1 ~ "1",
                              minchisq == chisq_i2 ~ "2",
                              minchisq == chisq_i3 ~ "3")) %>%
         mutate(whichmin = factor(whichmin, 
                                  levels = c("1, 2, 3", "2", "3", "1", 
                                             "2, 3", "1, 2", "1, 3"), 
                                  labels =c("All", "Reference-Indicator","Unit-Variance", "Integer",
                                            "RI & UV", "RI & I", "UV & I")))

which_diff %>% filter(same == "different") %>% group_by(whichdiff) %>% summarize(n = n() / 923)

# Add fancy header columns to table
addtorow3d <- list()
addtorow3d$pos <- list(0,0,0)
addtorow3d$command <- c("Best Fit & \\multicolumn{9}{c}{Proportion Sparse Indicators}\\\\",
                       "& \\multicolumn{3}{c}{Loadings: 0.4} & \\multicolumn{3}{c}{Loadings: 0.6} & \\multicolumn{3}{c}{Loadings: 0.8}\\\\",
                       #"& \\multicolumn{9}{c}{Proportion Sparse Indicators}\\\\",
                       "& 0.33 & 0.67 & 1.00 & 0.33 & 0.67 & 1.00 & 0.33 & 0.67 & 1.00\\\\")

tab_chisq3 <- which_diff %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag)%>%
  mutate(groupn = n()) %>%
  ungroup() %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, whichmin,.drop =FALSE) %>%
  summarize(pmin = n(),
            groupn = max(groupn, na.rm=T)) %>%
  mutate(pmin = abs(pmin/groupn)) %>%
  select(-groupn) %>%
  filter(resp_pattern == "sparse2" & num_ind == 6 & resp_options == 3) %>%
  ungroup() %>%
  select(-resp_pattern,-num_ind,-resp_options)

tab_chisq3_spec <-build_wider_spec(tab_chisq3, names_from= c("loading_mag","prop_sparse"), values_from = pmin)
tab_chisq3_spec <- arrange(tab_chisq3_spec, loading_mag, prop_sparse)

tab_chisq3d <- pivot_wider_spec(tab_chisq3, tab_chisq3_spec)


# Add fancy header columns to table
addtorow4d <- list()
addtorow4d$pos <- list(0,0,0)
addtorow4d$command <- c("Resp.Options & Best Fit & \\multicolumn{9}{c}{Proportion Sparse Indicators}\\\\",
                       "& & \\multicolumn{3}{c}{Loadings: 0.4} & \\multicolumn{3}{c}{Loadings: 0.6} & \\multicolumn{3}{c}{Loadings: 0.8}\\\\",
                       #"& \\multicolumn{9}{c}{Proportion Sparse Indicators}\\\\",
                       "& & 0.33 & 0.67 & 1.00 & 0.33 & 0.67 & 1.00 & 0.33 & 0.67 & 1.00\\\\")

tab_chisq4 <- which_diff %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag)%>%
  mutate(groupn = n()) %>%
  ungroup() %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, whichmin,.drop =FALSE) %>%
  summarize(pmin = n(),
            groupn = max(groupn,na.rm=T)) %>%
  mutate(pmin = abs(pmin/groupn)) %>%
  select(-groupn) %>%
  filter(resp_pattern == "sparse2" & num_ind == 3) %>%
  ungroup() %>%
  select(-resp_pattern) %>%
  drop_na() %>%
  select(resp_options,whichmin, loading_mag, prop_sparse, pmin)

tab_chisq4_spec <-build_wider_spec(tab_chisq4, names_from= c("loading_mag","prop_sparse"), values_from = pmin)
tab_chisq4_spec <- arrange(tab_chisq4_spec, loading_mag, prop_sparse)

tab_chisq4d <- pivot_wider_spec(tab_chisq4, tab_chisq4_spec) %>%
  arrange(resp_options, whichmin)

tab_chisq4d$resp_options <- ifelse(duplicated(tab_chisq4d$resp_options),"",tab_chisq4d$resp_options)

tab_chisq5 <- which_diff %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag)%>%
  mutate(groupn = n()) %>%
  ungroup() %>%
  group_by(resp_options, resp_pattern, num_ind, prop_sparse, loading_mag, whichmin,.drop =FALSE) %>%
  summarize(pmin = n(),
            groupn = max(groupn,na.rm=T)) %>%
  mutate(pmin = abs(pmin/groupn)) %>%
  select(-groupn) %>%
  filter(resp_pattern == "sparse2"& num_ind == 6) %>%
  ungroup() %>%
  select(-resp_pattern) %>%
  drop_na() %>%
  select(resp_options,whichmin, loading_mag, prop_sparse, pmin)

tab_chisq5_spec <-build_wider_spec(tab_chisq5, names_from= c("loading_mag","prop_sparse"), values_from = pmin)
tab_chisq5_spec <- arrange(tab_chisq5_spec, loading_mag, prop_sparse)

tab_chisq5d <- pivot_wider_spec(tab_chisq5, tab_chisq5_spec) %>%
  arrange(resp_options, whichmin)

tab_chisq5d$resp_options <- ifelse(duplicated(tab_chisq5d$resp_options),"",tab_chisq5d$resp_options)


save(tab_chisq1d, tab_chisq2d, tab_chisq3d, tab_chisq4d,tab_chisq5d,
     addtorow1d, addtorow2d, addtorow3d, addtorow4d, file = "figures/manuscript_revision_default_tables.RData")

