# Step 3: Run Simulation ----

# Prep ----
# Load packages
library(lavaan)
library(magrittr)
library(dplyr)
library(foreach)
library(parallel)

# Load Integer identification function
source("../altcoding.R")

# Load simulated data
sim_data <- readRDS("data/simdata.RDS")

# Set up run simulation function
run_sim <- function(iter_dat) {
  tryCatch(
    {
      num_ind <- (ncol(iter_dat) - 2) / 3
      
      indrange <- list(1:num_ind, (num_ind+1):(num_ind*2), (num_ind*2+1):(num_ind*3))
      
      # Set up estimation model
      f1_l <- paste0("x", indrange[[1]], collapse = "+")
      f2_l <- paste0("x", indrange[[2]], collapse = "+")
      f3_l <- paste0("x", indrange[[3]], collapse = "+")
      
      est_mod <- paste0('f1 =~ ',f1_l,'
                    f2 =~ ', f2_l,'
                    f3 =~ ', f3_l,
                        collapse = "")
      
      
      
      modsyn <- altcoding(est_mod, data = iter_dat, ordered = TRUE)
      
      ### Output: convergence + warnings
      fit1 <- lavaan(modsyn, data = iter_dat, ordered = TRUE, start = "simple")
      
      conv1 <- lavInspect(fit1, "converged")
      admis1 <- lavInspect(fit1, "post.check")
      
      fit2 <- cfa(est_mod, data = iter_dat, ordered = TRUE, start = "simple")
      
      conv2 <- lavInspect(fit2, "converged")
      admis2 <- lavInspect(fit2, "post.check")
      
      fit3 <- cfa(est_mod, data = iter_dat, ordered = TRUE, std.lv = T, start = "simple")
      
      conv3 <- lavInspect(fit3, "converged")
      admis3 <- lavInspect(fit3, "post.check")
      
      ### Output: fit
      fit1 <- lavaan(modsyn, data = iter_dat, ordered = TRUE, start = "simple", optim.force.converged = TRUE)
      fit2 <- cfa(est_mod, data = iter_dat, ordered = TRUE, start = "simple", optim.force.converged = TRUE)
      fit3 <- cfa(est_mod, data = iter_dat, ordered = TRUE, std.lv = T, start = "simple", optim.force.converged = TRUE)
      
      ### Output: fit
      fitm1 <- as.character(fitMeasures(fit1))[1:4]
      fitm2 <- as.character(fitMeasures(fit2))[1:4]
      fitm3 <- as.character(fitMeasures(fit3))[1:4]
      
      ### Output: combine
      condition_number <- iter_dat$condition_number[1]
      iter <- iter_dat$iter[1]
      
      res <- data.frame(condition_number, iter, conv1, admis1, 
                        fitm1[1],fitm1[2],fitm1[3],fitm1[4], 
                        conv2, admis2, 
                        fitm2[1],fitm2[2],fitm2[3],fitm2[4],
                        conv3, admis3, 
                        fitm3[1],fitm3[2],fitm3[3],fitm3[4],
                        NA)
      colnames(res) <- c("condition_number","iter",
                        "i1_conv", "i1_admis", "i1_npar", "i1_fmin", "i1_chisq", "i1_df",
                        "i2_conv", "i2_admis", "i2_npar", "i2_fmin", "i2_chisq", "i2_df",
                        "i3_conv", "i3_admis", "i3_npar", "i3_fmin", "i3_chisq", "i3_df",
                        "err")
      
      return(res)
    }, 
    error = function(e) {
      condition_number <- iter_dat$condition_number[1]
      iter <- iter_dat$iter[1]
      err <<- e
      err <- as.character(err)
      
      res <- data.frame(t(c(condition_number, iter, rep(NA, 18))), err)
      colnames(res) <- c("condition_number","iter",
                        "i1_conv", "i1_admis", "i1_npar", "i1_fmin", "i1_chisq", "i1_df",
                        "i2_conv", "i2_admis", "i2_npar", "i2_fmin", "i2_chisq", "i2_df",
                        "i3_conv", "i3_admis", "i3_npar", "i3_fmin", "i3_chisq", "i3_df",
                        "err")
      
      return(res)
    }
  )
  
  
}

# Run Simulation ----
# get cores for parallel processing (specified in simulation_execute.R)
n.cores <- n.cores

# Set up cluster, PSOCK is less of a memory hog for some reason
my.cluster <- parallel::makeCluster(n.cores, type = "PSOCK")
doParallel::registerDoParallel(cl = my.cluster)

sim_results <- foreach(i = sim_data,
                       .combine = "rbind",
                       .packages = c("lavaan"),
                       .noexport = c("simdat", "sim_conditions", "sim_iterations")) %dopar% {
                         
                         return(run_sim(iter_dat = i))
                       }
# Stop cluster
stopCluster(my.cluster)

# Save simulation output
saveRDS(sim_results, file = "results/simresults_simple.RDS")
